// Mobile menu
const menuBtn = document.getElementById('menuBtn');
const menu = document.getElementById('menu');
// guard menu in case the element is missing
menuBtn?.addEventListener('click', () => menu?.classList.toggle('show'));

// Smooth scroll for in-page links
document.querySelectorAll('a[href^="#"]').forEach(a => {
  a.addEventListener('click', e => {
    const id = a.getAttribute('href');
    if (id.length > 1) {
      e.preventDefault();
      document.querySelector(id)?.scrollIntoView({ behavior: 'smooth', block: 'start' });
      menu?.classList.remove('show');
    }
  });
});

// Contact form (no backend) — opens email with message prefilled
document.getElementById('contactForm')?.addEventListener('submit', function (e) {
  e.preventDefault();
  const data = new FormData(this);
  const subject = encodeURIComponent('Enquiry from School Website');
  const body = encodeURIComponent(
    `Name: ${data.get('name')}\nEmail: ${data.get('email')}\n\nMessage:\n${data.get('message')}`
  );
  window.location.href = `mailto:peakacademyedu@gmail.com?subject=${subject}&body=${body}`;
  const helpEl = document.getElementById('formHelp');
  if (helpEl) helpEl.textContent = 'Opening your email app…';
});

// Results demo (show card on submit)
document.getElementById('resultForm')?.addEventListener('submit', function (e) {
  e.preventDefault();
  checkResult();
});


// Year auto-update (guard if element missing)
const yearEl = document.getElementById('year');
if (yearEl) yearEl.textContent = new Date().getFullYear();

function printDiv(divId) {
  const el = document.getElementById(divId);
  if (!el) {
    alert('Nothing to print: element not found.');
    return;
  }
  const content = el.innerHTML;
  const win = window.open('', '', 'width=800,height=600');
  if (!win) {
    alert('Unable to open print window (popup blocked).');
    return;
  }
  win.document.write(`
    <html>
      <head>
        <title>Print</title>
        <style>
          body { font-family: Arial, sans-serif; padding: 20px; }
          table { width: 100%; border-collapse: collapse; }
          th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        </style>
      </head>
      <body>${content}</body>
    </html>
  `);
  win.document.close();
  win.print();
}

function printResult() {
  const resultEl = document.getElementById('resultContent');
  if (!resultEl) {
    alert('No result content to print.');
    return;
  }
  const resultDiv = resultEl.innerHTML;
  const newWin = window.open('', '_blank');
  if (!newWin) {
    alert('Unable to open print window (popup blocked).');
    return;
  }
  newWin.document.write(`
    <html>
      <head>
        <title>Student Result</title>
        <style>
          body { font-family: Arial, sans-serif; }
          table { border-collapse: collapse; width: 100%; }
          th, td { border: 1px solid #000; padding: 8px; text-align: left; }
        </style>
      </head>
      <body>
        <div style="text-align:center">
          <h2>PEAK ACADEMY</h2>
          <h3>Term Result Sheet</h3>
        </div>
        ${resultDiv}
      </body>
    </html>
  `);
  newWin.document.close();
  newWin.print();
}


async function checkResult() {
  const admissionNo = document.getElementById("admissionNo").value;
  const term = document.getElementById("term").value;
  const session = document.getElementById("session").value;

  // Guard: ensure inputs exist
  const admEl = document.getElementById('admissionNo');
  const termEl = document.getElementById('term');
  const sessEl = document.getElementById('session');
  if (!admEl || !termEl || !sessEl) {
    alert('Result form elements are missing.');
    return;
  }

  // Fetch result data (with error handling)
  let results = [];
  try {
    const res = await fetch('results.json');
    if (!res.ok) throw new Error('Failed to load results.json');
    results = await res.json();
  } catch (err) {
    console.error('Error fetching results:', err);
    alert('Unable to load results data. Please try again later.');
    return;
  }

  // Search for student
  const student = results.find(r =>
    r.admissionNo === admissionNo &&
    r.term === term &&
    r.session === session
  );

  if (student) {
    let subjectsHTML = "";
    student.subjects.forEach(sub => {
      subjectsHTML += `<tr><td>${sub.name}</td><td>${sub.score}</td></tr>`;
    });

    document.getElementById("resultContent").innerHTML = `
      <h3>${student.name} (${student.admissionNo})</h3>
      <p>Class: ${student.class} | Term: ${student.term} | Session: ${student.session}</p>
      <table border="1" cellpadding="5" cellspacing="0">
        <tr><th>Subject</th><th>Score</th></tr>
        ${subjectsHTML}
      </table>
      <p><strong>Average:</strong> ${student.average}</p>
      <p><strong>Remark:</strong> ${student.remark}</p>
    `;
    document.getElementById("resultCard").style.display = "block";
  } else {
    alert("❌ No result found for this student. Please check Admission No, Term, and Session.");
    document.getElementById("resultContent").innerHTML =
      "<p style='color:red'>❌ No result found. Check Admission No/Term/Session</p>";
    document.getElementById("resultCard").style.display = "block";
  }
}

// Registration Popup Functions
document.getElementById('registerNowBtn')?.addEventListener('click', function() {
  const popup = document.getElementById('registrationPopup');
  if (popup) popup.style.display = 'flex';
});

function closeRegistrationPopup() {
  const popup = document.getElementById('registrationPopup');
  if (popup) popup.style.display = 'none';
}

// Close popup when clicking outside the content
document.getElementById('registrationPopup')?.addEventListener('click', function(e) {
  if (e.target === this) closeRegistrationPopup();
});

// Student Login Toggle
document.getElementById('loginToggleBtn')?.addEventListener('click', function() {
  const form = document.getElementById('studentLoginForm');
  if (form) {
    form.style.display = form.style.display === 'none' ? 'block' : 'none';
  }
});

// Student Login Form Submission
document.getElementById('studentLoginForm')?.addEventListener('submit', function(e) {
  e.preventDefault();
  
  const studentName = document.getElementById('studentName').value.trim();
  const studentPassword = document.getElementById('studentPassword').value.trim();
  
  // Simple validation
  if (!studentName || !studentPassword) {
    alert('Please enter both name and password.');
    return;
  }
  
  // Store credentials in sessionStorage (simple approach)
  sessionStorage.setItem('studentName', studentName);
  sessionStorage.setItem('studentPassword', studentPassword);
  
  // Redirect to student portal
  window.location.href = '/student.html';
});